/*
$Id: ow_presence.c,v 1.31 2009/07/04 16:53:07 alfille Exp $
    OWFS -- One-Wire filesystem
    OWHTTPD -- One-Wire Web Server
    Written 2003 Paul H Alfille
    email: palfille@earthlink.net
    Released under the GPL
    See the header file: ow.h for full attribution
    1wire/iButton system from Dallas Semiconductor
*/

/* General Device File format:
    This device file corresponds to a specific 1wire/iButton chip type
    ( or a closely related family of chips )

    The connection to the larger program is through the "device" data structure,
      which must be declared in the acompanying header file.

    The device structure holds the
      family code,
      name,
      device type (chip, interface or pseudo)
      number of properties,
      list of property structures, called "filetype".

    Each filetype structure holds the
      name,
      estimated length (in bytes),
      aggregate structure pointer,
      data format,
      read function,
      write funtion,
      generic data pointer

    The aggregate structure, is present for properties that several members
    (e.g. pages of memory or entries in a temperature log. It holds:
      number of elements
      whether the members are lettered or numbered
      whether the elements are stored together and split, or separately and joined
*/

#include <config.h>
#include "owfs_config.h"
#include "ow_xxxx.h"
#include "ow_counters.h"
#include "ow_connection.h"

/* ------- Prototypes ------------ */
static int CheckPresence_low(const struct parsedname *pn);
static int CheckThisConnection(int bus_nr, const struct parsedname *pn) ;

/* ------- Functions ------------ */

/* Check if device exists -- >=0 yes, -1 no */
int CheckPresence(struct parsedname *pn)
{
	int bus_nr;
	
	if (NotRealDir(pn)) {
		return 0;
	}
	
	if ((pn->selected_device == DeviceSimultaneous)
		|| (pn->selected_device == DeviceThermostat)) {
		return 0;
	}
	
	/* If set, already found bus. */
	/* Use UnsetKnownBus to clear and allow a new search */
	if (KnownBus(pn)) {
		return pn->known_bus->index;
	}
	
	if (Cache_Get_Device(&bus_nr, pn) == 0) {
		LEVEL_DEBUG("Found device on bus %d\n",bus_nr);
		SetKnownBus(bus_nr, pn);
		return bus_nr;
	}
	
	LEVEL_DETAIL("Checking presence of %s\n", SAFESTRING(pn->path));
	
	if ( Inbound_Control.active == 0 ) { // No adapters
		return -1 ;
	}
	bus_nr = CheckPresence_low(pn);	// check only allocated inbound connections
	if (bus_nr >= 0) {
		SetKnownBus(bus_nr, pn);
		return bus_nr;
	}
	UnsetKnownBus(pn);
	return -1;
}

/* See if a cached location is accurate -- called with "Known Bus" set */
int ReCheckPresence(struct parsedname *pn)
{
	int bus_nr;
	
	if (NotRealDir(pn)) {
		return 0;
	}
	
	if ((pn->selected_device == DeviceSimultaneous)
		|| (pn->selected_device == DeviceThermostat)) {
		return 0;
	}
	
	if (KnownBus(pn)) {
		if ( CheckThisConnection(pn->known_bus->index,pn) >= 0 ) {
			return pn->known_bus->index ;
		}
	}
	
	if (Cache_Get_Device(&bus_nr, pn) == 0) {
		LEVEL_DEBUG("Found device on bus %d\n",bus_nr);
		if ( CheckThisConnection(bus_nr,pn) >= 0 ) {
			SetKnownBus(bus_nr, pn);
			return bus_nr ;
		}
	}
	
	UnsetKnownBus(pn);
	Cache_Del_Device(pn) ;
	return CheckPresence(pn);
}

/* Check if device exists -- -1 no, >=0 yes (bus number) */
/* lower level, cycle through the devices */
#if OW_MT

struct checkpresence_struct {
	struct connection_in *in;
	const struct parsedname *pn;
	int bus_nr;
};

static void * CheckPresence_callback(void * v)
{
	pthread_t thread;
	int threadbad = 1;
	struct checkpresence_struct * cps = (struct checkpresence_struct *) v ;
	struct checkpresence_struct next_cps = { cps->in->next, cps->pn, -ENOENT };
	
	threadbad = (next_cps.in == NULL)
	|| pthread_create(&thread, NULL, CheckPresence_callback, (void *) (&next_cps));
	
	cps->bus_nr = CheckThisConnection( cps->in->index, cps->pn ) ;
	
	if (threadbad == 0) {		/* was a thread created? */
		void *vv;
		if (pthread_join(thread, &vv)==0) {
			if ( next_cps.bus_nr >= 0 ) {
				cps->bus_nr = next_cps.bus_nr ;
			}
		}
	}
	return NULL ;
}

static int CheckPresence_low(const struct parsedname *pn)
{
	struct checkpresence_struct cps = { Inbound_Control.head , pn, -ENOENT };
		
	if ( cps.in != NULL ) {
		CheckPresence_callback( (void *) (&cps) ) ;
	}
	return cps.bus_nr;
}

#else							/* OW_MT */

static int CheckPresence_low(const struct parsedname *pn)
{
	struct connection_in * in ;
	
	for ( in=Inbound_Control.head ; in ; in=in->next ) {
		int bus_nr = CheckThisConnection( in->index, pn ) ;
		if ( bus_nr >= 0 ) {
			return bus_nr ;
		}
	}
	return -ENOENT;				// no success
}
#endif							/* OW_MT */

int FS_present(struct one_wire_query *owq)
{
	struct parsedname *pn = PN(owq);

	if (NotRealDir(pn) || pn->selected_device == DeviceSimultaneous || pn->selected_device == DeviceThermostat) {
		OWQ_Y(owq) = 1;
	} else if (get_busmode(pn->selected_connection) == bus_fake) {
		OWQ_Y(owq) = 1;
	} else if (get_busmode(pn->selected_connection) == bus_tester) {
		OWQ_Y(owq) = 1;
	} else if (get_busmode(pn->selected_connection) == bus_mock) {
		OWQ_Y(owq) = 1;
	} else {
		struct transaction_log t[] = {
			TRXN_NVERIFY,
			TRXN_END,
		};
		OWQ_Y(owq) = BUS_transaction(t, pn) ? 0 : 1;
	}
	return 0;
}

static int CheckThisConnection(int bus_nr, const struct parsedname *pn)
{
	struct parsedname s_pn_copy;
	struct parsedname * pn_copy = &s_pn_copy ;
	struct connection_in * in = find_connection_in(bus_nr) ;

	if ( in == NULL ) {
		return -ENOENT ;
	}
	
	memcpy(pn_copy, pn, sizeof(struct parsedname));	// shallow copy
	pn_copy->selected_connection = in;
	
	if (TestConnection(pn_copy)) {	// reconnect successful?
		return -ECONNABORTED;
	} else if (BusIsServer(in)) {
		//printf("CheckPresence_low: call ServerPresence\n");
		if (ServerPresence(pn_copy) >= 0) {
			/* Device was found on this in-device, return it's index */
			LEVEL_DEBUG("Presence found on server bus %s\n",SAFESTRING(in->name)) ;
			Cache_Add_Device(in->index,pn_copy->sn) ; // add or update cache */
			return in->index;
		}
		//printf("CheckPresence_low: ServerPresence(%s) pn->selected_connection->index=%d ret=%d\n", pn->path, pn->selected_connection->index, ret);
	} else if ( (get_busmode(in) == bus_fake) || (get_busmode(in) == bus_tester) || (get_busmode(in) == bus_mock) ) {
		if ( DirblobSearch(pn_copy->sn, &(in->main)) >= 0 ) {
			LEVEL_DEBUG("Presence found on fake-like bus %s\n",SAFESTRING(in->name)) ;
			return in->index;
		}
	} else {
		struct transaction_log t[] = {
			TRXN_NVERIFY,
			TRXN_END,
		};
		/* this can only be done on local buses */
		if (BUS_transaction(t, pn_copy) == 0 ) {
			/* Device was found on this in-device, return it's index */
			LEVEL_DEBUG("Presence found on local bus %s\n",SAFESTRING(in->name)) ;
			Cache_Add_Device(in->index,pn_copy->sn) ; // add or update cache */
			return in->index ;
		}
	}
	LEVEL_DEBUG("Presence NOT found on bus %s\n",SAFESTRING(in->name)) ;
	return -1 ;
}

