/*
 * $Id: temploggerd.c,v 1.12 2007/10/12 12:46:48 d1mag Exp $
 *
 * temploggerd -- graph/log utility written for OWFS
 * Written 2005 Christian Magnusson
 * email: mag@mag.cx
 * Released under the GPL
*/
#include <features.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <dirent.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <ctype.h>
#include <stdarg.h>

#include <globals.h>
#include <rrdfile.h>
#include <sensor.h>
#include <util.h>
#include <config_vars.h>
#include <html.h>
#include <scan_bus.h>

unsigned char save_all_sensors = 0;
unsigned char foreground = 0;
unsigned char generate_templates = 1;
#ifdef __UCLIBC__
/* We usually don't have enough memory on embedded systems */
unsigned char auto_add_sensors = 0;
#else
unsigned char auto_add_sensors = 1;
#endif
int print_rrd_commands = 0;

int loopdelay = 60;
char pid_file[FILENAME_LEN+1];
char device_path[FILENAME_LEN+1];
char uncached_device_path[FILENAME_LEN+1];


void usage(char **argv)
{
  printf("Usage: %s [-v] [-c config_file] [-P pid_file] [-d] [-s] [-f]\n", argv[0]);
  printf("  -v\tPrint version\n");
  printf("  -c\tSet config-file (Currently using [%s])\n", config_file);
  printf("  -P\tSave process id to file\n");
  printf("  -p\tPrint default settings\n");
  printf("  -d\tPrint rrdtool commands\n");
#ifdef __UCLIBC__
  printf("  -a\tEnable AutoAdd new sensors to rrd-files\n");
  printf("  -b\tDisable AutoAdd new sensors to rrd-files (default disabled)\n");
#else
  printf("  -a\tEnable AutoAdd new sensors to rrd-files (default enabled)\n");
  printf("  -b\tDisable AutoAdd new sensors to rrd-files\n");
#endif
  printf("  -s\tSave ALL defined sensors (even if not on 1-wire bus)\n");
  printf("  -f\tStart in foreground (background as default)\n");
  exit(0);
}

int save_pidfile(void)
{
  FILE *fp;
  int rc;
  if(!pid_file[0]) return 0;
  fp = fopen(pid_file, "w");
  if(fp) {
    rc = fprintf(fp, "%d", getpid());
    fclose(fp);
  }
  if(!fp || (rc<0)) {
    printf("Error saving pid_file to [%s]\n", pid_file);
    return -1;
  }
  return 1;
}

void print_version(void)
{
  printf("Version: %s\n", VERSION);
  exit(0);
}

static void sighup_handler(int sig) {
  (void) sig;
  generate_templates = 1;
  printf("Will scan 1-wire bus and generate new templates\n");
}

void set_signal_handlers( void ) {
  struct sigaction sa;

  memset(&sa, 0, sizeof(struct sigaction));
  sigemptyset(&(sa.sa_mask));
  sa.sa_flags = SA_RESTART;
  sa.sa_handler = sighup_handler;

  if ((sigaction(SIGHUP,  &sa, NULL) == -1)) {
    printf("Cannot setup signal handlers");
    exit(0);
  }
}

int restore_files(int force)
{
  char cmd[2*FILENAME_LEN+32];
  char backupdir[FILENAME_LEN+1];
  char wwwdir[FILENAME_LEN+1];
  struct stat bst;
  struct stat wst;
  int rc;

  if(!backup_dir || !backup_dir[0]) return 0;

  sprintf(backupdir, "%s/%s", backup_dir, base_dir);

  if(stat(backupdir, &bst) < 0) {
    printf("Backup doesn't exist in [%s]\n", backup_dir);
    return 0;
  }
  if(!S_ISDIR(bst.st_mode)) {
    printf("Backup isn't a directory [%s]\n", backup_dir);
    return 0;
  }
  
  sprintf(wwwdir, "%s/%s", www_dir, base_dir);
  if(stat(wwwdir, &wst) < 0) {
    printf("Directory doesn't exist [%s]. Create it.\n", wwwdir);
    sprintf(cmd, "mkdir -p %s", wwwdir);
    rc = system(cmd);
    if(rc < 0) {
      printf("restore_files: mkdir failed [%s]\n", cmd);
    }
    force = 1;
  }
  if((stat(wwwdir, &wst) < 0) || !S_ISDIR(wst.st_mode)) {
    printf("Directory doesn't exist [%s]\n", wwwdir);
    return 0;
  }

  if(force || (bst.st_mtime > wst.st_mtime)) {
    printf("Restore files from [%s/%s/*] to [%s]\n", backup_dir, base_dir, wwwdir);
    sprintf(cmd, "cp -Rpf %s/%s/* %s", backup_dir, base_dir, wwwdir);
    printf("run [%s]\n", cmd);
    rc = system(cmd);
    if(rc < 0) {
      printf("restore_files: cp failed [%s]\n", cmd);
    }
  }
  return 0;
}

int backup_files(void)
{
  char cmd[2*FILENAME_LEN+32];
  char wwwdir[FILENAME_LEN+1];
  struct stat wst;
  struct stat bst;
  int rc;

  if(!backup_dir || !backup_dir[0]) return 0;

  sprintf(wwwdir, "%s/%s", www_dir, base_dir);
  if((stat(wwwdir, &wst) < 0) || !S_ISDIR(wst.st_mode)) {
    printf("[%s] is not a directory\n", wwwdir);
    return 0;
  }

  if(stat(backup_dir, &bst) < 0) {
    printf("Backup-directory doesn't exist [%s]. Create it.\n", backup_dir);
    sprintf(cmd, "mkdir -p %s", backup_dir);
    rc = system(cmd);
    if(rc < 0) {
      printf("backup_files: mkdir failed [%s]\n", cmd);
    }
  } else if(!S_ISDIR(bst.st_mode)) {
    printf("Backup-path is not a directory [%s]\n", backup_dir);
    return 0;
  }
  
  printf("Backup files from [%s] to [%s]\n", wwwdir, backup_dir);
  sprintf(cmd, "cp -Rpf %s %s", wwwdir, backup_dir);
  printf("run [%s]\n", cmd);
  rc = system(cmd);
  if(rc < 0) {
    printf("backup_files: cp failed [%s]\n", cmd);
  }
  return 0;
}

#define CMD_DEFAULT_LEN 2048

int check_and_create_rrd_files(void)
{
  int cmd_len;
  char *cmd;
  struct sensor *s;
  struct rrd_file *r;
  int i, len, rc;
  struct stat st;
  char rrdfile[FILENAME_LEN+1];
  char rra[4][80];
  char rrdid[FILENAME_LEN+1];

  r = rrd_files;
  while(r) {
#if 0
    printf("rrdfile [%s] nr_sensors=%d\n", r->name, r->nr_sensors);
    s = sensors;
    while(s) {
      if(s->rrdfile == r) printf(" %s[%s]", (s->ignored?"IGN":""), s->id);
      s = s->next;
    }
    printf("\n");
#endif
    if(r->nr_sensors == 0) {
      //printf("no sensors in %s. Skip it.\n", r->name);
      r = r->next;
      continue;
    }

    sprintf(rrdfile, "%s/%s", rrdfile_path, r->name);
    if(stat(rrdfile, &st) < 0) {
      printf("Warning: [%s] doesn't exist. Creating it now!\n", rrdfile);
      cmd_len = CMD_DEFAULT_LEN;
      if(!(cmd = malloc(cmd_len))) {
	printf("Error malloc cmd buf\n");
	exit(0);
      }
      len = sprintf(cmd, "%s create %s --step %d ", rrdtool, rrdfile, loopdelay);
      
      for(i=0; i<4; i++) rra[i][0] = '\000';
      sprintf(rra[0], "%d:%d ", 1, 2 * 24 * 3600 / loopdelay);
      // * 14 days of values at 1/2 hour resolution
      sprintf(rra[1], "%d:%d ", 30 * 60 / loopdelay, 14 * 24 * 2);
      // * 3 month at 2 hours resolution
      sprintf(rra[2], "%d:%d ", 2 * 3600 / loopdelay, 2 * 31 * 12);
      // * 2 years at 1 day resolution
      sprintf(rra[3], "%d:%d ", 24 * 3600 / loopdelay, 2 * 365);
      s = sensors;
      while(s) {
	if(s->rrdfile != r) {
	  s = s->next;
	  continue;
	}
	if((save_all_sensors || s->file_found_on_bus) && !s->ignored) {
	  char *rrdtype[3] = { "GAUGE", "COUNTER", "DERIVE" };
	  rrd_sensid(s, rrdid);
	  //printf("create [%s]\n", rrdid);
	  len += sprintf(&cmd[len], " DS:%s:%s:%d:%d:%d", rrdid, rrdtype[s->rrdtype], 2*loopdelay, s->minval, s->maxval);
	}
	if(len > (cmd_len>>1)) {
	  cmd_len <<= 1;
	  //printf("Resize cmd to %d (len=%d)\n", cmd_len, len);
	  cmd = realloc(cmd, cmd_len);
	  if(!cmd) {
	    printf("Failed to realloc cmd to %d\n", cmd_len);
	    break;
	  }
	}
	s = s->next;
      }
      for(i=0; i<4; i++) {
	len += sprintf(&cmd[len], " RRA:AVERAGE:0.5:%s", rra[i]);
	//strcat(cmd, tmp);
      }
      for(i=0; i<4; i++) {
	len += sprintf(&cmd[len], " RRA:MIN:0.5:%s", rra[i]);
	//strcat(cmd, tmp);
      }
      for(i=0; i<4; i++) {
	len += sprintf(&cmd[len], " RRA:MAX:0.5:%s", rra[i]);
	//strcat(cmd, tmp);
      }
      if(cmd) {
	if(print_rrd_commands) printf("execute: %s\n", cmd);
	rc = system(cmd);
	free(cmd);
	if(rc < 0) {
	  printf("Error creating rrdfile %s\n", rrdfile);
	  exit(0);
	}
      }
    }
    r = r->next;
  }
  return 0;
}

int main(int argc, char **argv)
{
  char *cmd;
  char tmp[256];
  char rrdfile[FILENAME_LEN+1];
  struct sensor *s;
  struct rrd_file *r;
  struct timeval last_update;
  int nr_errors;
  int nr_sensors;
  unsigned char found_fuse = 0;
  struct stat st;
  int arg, len;
  int cmd_len;
  int tot_waited;
  time_t backup_next = 0;
  time_t next_scanbus = 0;
  time_t tnow;

  last_update.tv_sec = last_update.tv_usec = 0;
  pid_file[0] = '\000';

  set_defaults();

  arg = 1;
  while(arg < argc) {
    if(argv[arg][0] == '-') {
      switch(argv[arg][1]) {
      case 'a':
	auto_add_sensors = 1;
	break;
      case 'b':
	auto_add_sensors = 0;
	break;
      case 'c':
	if(argc <= ++arg) usage(argv);
	strncpy(config_file, argv[arg], FILENAME_LEN);
	config_file[FILENAME_LEN] = '\000';
	break;
      case 'v':
	print_version();
	break;
      case 'p':
	read_config_file(0);
	print_defaults();
	exit(0);
	break;
      case 'P':
	if(argc <= ++arg) usage(argv);
	strncpy(pid_file, argv[arg], FILENAME_LEN);
	pid_file[FILENAME_LEN] = '\000';
	break;
      case 's':
	save_all_sensors = 1;
	break;
      case 'd':
	print_rrd_commands = 1;
	break;
      case 'f':
	foreground = 1;
	break;
      default:
	usage(argv);
      }
    } else {
      usage(argv);
    }
    arg++;
  }

  if(print_rrd_commands) {
    foreground = 1;
  }

  initiate_colors();

  //printf("Read config-file first time\n");
  if(read_config_file(0) < 0) {
    exit(0);
  }

  if(!(found_fuse = find_fusedev(owfs_dir, device_path, uncached_device_path))) {
    if(owfs_dir[0]) {
      printf("Can't find owfs/fuse directory at [%s]\n", owfs_dir);
    } else {
      printf("Can't find owfs/fuse directory in /proc/mounts\n");
    }
    exit(0);
  }

  nr_sensors = scan_bus(uncached_device_path);
  if(new_device_found) {
    new_device_found = 0;
    generate_templates = 1;
  }

  if(!save_all_sensors && (nr_sensors == 0)) {
    printf("No sensors found on the 1-wire bus\n");
  }

  /*
    Read config-file and perhaps change name on the devices, or ignore them
  */
  if(read_config_file(1) < 0) {
    exit(0);
  }

  nr_sensors = 0;
  s = sensors;
  while(s) {
    if(!s->ignored) {
      if(!is_tempsensor(sensor_id(s)) && !s->full_path[0]) {
	//printf("ignore sensor [%s] [%s] [%s]\n", s->id, s->datafile, s->full_path);
	s->ignored = 1;
      }
      if((save_all_sensors || s->file_found_on_bus)) {
	nr_sensors++;
	if(s->rrdfile)
	  s->rrdfile->nr_sensors++;
	//else printf("Warning: [%s] is not assigned to any rrd-file\n", s->id);
      }
    }
    s = s->next;
  }
  if(nr_sensors == 0) {
    printf("No sensors found or used.\n");
  }

  s = sensors;
  while(s) {
    if(!s->config_sensor && !s->file_found_on_bus) {
      s->ignored = 1;
      printf("Warning: Ignoring unknown sensor [%s]\n", s->id);
    }
    if(s->config_sensor && !s->file_found_on_bus) {
      if(!s->ignored) printf("Warning: Defined sensor in config-file not found [%s] [%s]\n", s->id, s->datafile);
    }

    //printf("sens %s  rrd-file %s\n", s->id, s->rrdfile->name);
    r = s->rrdfile;
    if(r) {
      if(!r->tmpl_all[0]) {
	printf("Warning: No all-template for (%s), setting default [%s]\n", r->name, tmpl_all_default);
	strcpy(r->tmpl_all, tmpl_all_default);
      }
      if(!r->tmpl_sensor[0]) {
	printf("Warning: No sensor-template for (%s), setting default [%s]\n", r->name, tmpl_sensor_default);
	strcpy(r->tmpl_sensor, tmpl_sensor_default);
      }
      if(!r->description[0]) {
	printf("Warning: No description for (%s), setting default [%s]\n", r->name, r->name);
	strcpy(r->description, r->name);
      }
    } else {
      if(!s->ignored) {
	printf("Error: [%s] missing rrd-file and NOT ignored!\n", s->id);
	print_sensor(s);
	exit(0);
      }
    }
    s = s->next;
  }


  umask(0);

  if(stat(rrdcgi_file, &st) < 0) {
    printf("warning: Can't find rrdcgi at [%s]\n", rrdcgi_file);
  }
  if(stat(www_dir, &st) < 0) {
    printf("Warning: www_dir doesn't exist at [%s]\n", www_dir);
    if(mkdir(www_dir, 0777) < 0) {
      printf("Error creating directory [%s]\n", www_dir);
      exit(0);
    }
  }
  sprintf(tmp, "%s%s", www_dir, base_dir);
  if(stat(tmp, &st) < 0) {
    printf("Warning: base_dir doesn't exist at [%s]\n", tmp);
    if(mkdir(tmp, 0777) < 0) {
      printf("Error creating directory [%s]\n", tmp);
      exit(0);
    }
  }
  sprintf(tmp, "%s%s/%s", www_dir, base_dir, rrdimg_path);
  if(stat(tmp, &st) < 0) {
    printf("Warning: rrdimg_dir doesn't exist at [%s]\n", tmp);
    if(mkdir(tmp, 0777) < 0) {
      printf("Error creating directory [%s]\n", tmp);
      exit(0);
    }
  }

  if(stat(cgi_dir, &st) < 0) {
    printf("Warning: cgi-dir doesn't exist at [%s]\n", cgi_dir);
    if(mkdir(cgi_dir, 0777) < 0) {
      printf("Error creating directory [%s]\n", cgi_dir);
      exit(0);
    }
  }

  check_and_create_rrd_files();

  if(!foreground) {
#if defined(__UCLIBC__) && !(defined(__UCLIBC_HAS_MMU__) || defined(__ARCH_HAS_MMU__))
    printf("Warning: daemon() doesn't exist for this CPU.\n");
#else
    if(daemon(1,0) == -1) {
      printf("daemon() error\n");
      exit(0);
    }
#endif
  }

  set_signal_handlers();

  save_pidfile();


  /* read the rrd-files and see which DS exists in them */
  r = rrd_files;
  while(r) {
    cleanup_rrd_ds(r);
    read_rrd_file(r);
    r = r->next;
  }

  tot_waited = loopdelay;
  nr_errors = 0;
  backup_next = 0;
  restore_files(0);
  backup_next = time(NULL) + _backup_freq;

  while(1) {
    if(_backup_freq>0) {
      tnow = time(NULL);
      if(backup_next < tnow) {
	backup_files();
	backup_next = tnow + _backup_freq;
      }
    }
    if(generate_templates) {
      generate_templates = 0;
      if(read_config_file(1) < 0) {
	fprintf(stderr, "Error reading config-file\n");
      }
      nr_sensors = scan_bus(uncached_device_path);
      new_device_found = 0;
      update_sensors(device_path);
      create_webpage_index();
      copy_images_dir();
      s = sensors;
      while(s) {
	create_webpage_sensor(s);
	s = s->next;
      }
    }
    /* This delay will not be exactly "loopdelay" seconds, but I don't
       think it makes any difference */
    my_delay(1000);
    if(++tot_waited < loopdelay) {
      continue;
    }
    tot_waited = 0;

    if(!(found_fuse = find_fusedev(owfs_dir, device_path, uncached_device_path))) {
      if(owfs_dir[0]) {
	printf("Can't find owfs/fuse directory at [%s]\n", owfs_dir);
      } else {
	printf("Can't find owfs/fuse directory in /proc/mounts\n");
      }
      nr_errors++;
      continue;
    }

    tnow = time(NULL);
    if(tnow > next_scanbus) {
      /* scan bus for new devices every 5 minutes */
      nr_sensors = scan_bus(uncached_device_path);
      if(new_device_found) {
	new_device_found = 0;
	generate_templates = 1;
      }
      //printf("scan_bus ret=%d\n", nr_sensors);
      next_scanbus = tnow + 300;
    }
    r = rrd_files;
    while(r) {
      r->nr_sensors = 0;
      r = r->next;
    }
    nr_sensors = 0;
    s = sensors;
    while(s) {
      if(!s->ignored) {
	if(!is_tempsensor(sensor_id(s)) && !s->full_path[0]) {
	  //printf("ignore sensor [%s] [%s] [%s]\n", s->id, s->datafile, s->full_path);
	  s->ignored = 1;
	}
	if(save_all_sensors || s->file_found_on_bus) {
	  nr_sensors++;
	  if(s->rrdfile)
	    s->rrdfile->nr_sensors++;
	  //else printf("Warning: [%s] is not assigned to any rrd-file\n", s->id);
	}
      }
      s = s->next;
    }
    //printf("nr_sensors=%d\n", nr_sensors);

    update_sensors(device_path);

    if(nr_avail_sensors() == 0) {
      //printf("No sensors available anymore.\n");
      nr_errors++;
      continue;
    }
    cmd_len = CMD_DEFAULT_LEN;
    if(!(cmd = malloc(cmd_len))) {
      printf("Error malloc cmd buff2\n");
      exit(0);
    }

    r = rrd_files;
    while(r) {
      if(r->nr_sensors == 0) {
	r = r->next;
	continue;
      }
      
      sprintf(rrdfile, "%s/%s", rrdfile_path, r->name);

      len = sprintf(cmd, "%s", rrdtool);
      len += sprintf(&cmd[len], " 2>&1");
      len += sprintf(&cmd[len], " update %s --template ", rrdfile);

      s = sensors;
      while(s) {
	if(s->rrdfile != r) {
	  s = s->next;
	  continue;
	}
	if((save_all_sensors || s->file_found_on_bus) && !s->ignored) {
	  char rrdid[FILENAME_LEN+1];
	  rrd_sensid(s, rrdid);

	  if(!find_rrd_ds(r, rrdid)) {
	    int rc;
	    char cmd_add[2*FILENAME_LEN+1];
	    printf("Warning: Sensor [%s] [%s] doesn't exist in rrd-file\n", s->id, s->location);
	    if(auto_add_sensors && add_new_sensor_script[0]) {
	      sprintf(cmd_add, "%s %s %s/%s", add_new_sensor_script, rrdid, rrdfile_path, r->name);
	      if(print_rrd_commands) printf("Execute: [%s]\n", cmd_add);
	      rc = system(cmd_add);
	      if(rc == 0) {
		printf("DS-name [%s] is added to rrd-file\n", rrdid);
		add_rrd_ds(r, rrdid, 1);
		generate_templates = 1;
	      } else {
		printf("Failed to add sensor rc=%d cmd=[%s]\n", rc, cmd_add);
		// make sure it's ignored in the future if we can't add it
		s->ignored = 1;
		generate_templates = 1;
	      }
	    } else {
	      printf("DS-name [%s] is not added automatically. Ignoring it now\n", rrdid);
	      s->ignored = 1;
	      generate_templates = 1;
	    }

	  }
	  if(!s->ignored) len += sprintf(&cmd[len], "%s:", rrdid);
	}
	s = s->next;
      }
      len += sprintf(&cmd[len-1], " N:");
      len--;
      s = sensors;
      while(s) {
	if((s->rrdfile != r) || (s->rrdfile->nr_sensors==0)) {
	  s = s->next;
	  continue;
	}
	if((save_all_sensors || s->file_found_on_bus) && !s->ignored) {
	  if(s->avail) {
	    if(s->rrdtype == 0)
	      len += sprintf(&cmd[len], "%.2f:", s->last_value); // GAUGE
	    else
	      len += sprintf(&cmd[len], "%ld:", (long)s->last_value); // COUNTER,DERIVE
	  } else
	    len += sprintf(&cmd[len], "U:");
	}
	if(len > (cmd_len>>1)) {
	  cmd_len <<= 1;
	  //printf("Resize cmd to %d (len=%d)\n", cmd_len, len);
	  if(!(cmd = realloc(cmd, cmd_len))) {
	    printf("Failed to realloc cmd to %d\n", cmd_len);
	    break;
	  }
	}
	s = s->next;
      }

      if(cmd) {
	len += sprintf(&cmd[len-1], " ");
	len--;
	if(print_rrd_commands) printf("execute: %s\n", cmd);
      
#if 1
	if(system(cmd) < 0) {
	  printf("Error updating rrdfile %s\n", rrdfile);
	}
#else
	{
	  FILE *fp;
	  char *t;
	  char tmp[ROW_LEN+1];
	  if(!(fp = popen(cmd, "r"))) {
	    printf("error popen()\n");
	    break;
	  }
	  while(!feof(fp)) {
	    if(!(t = fgets(tmp, ROW_LEN, fp))) {
	      break;
	    }
	    printf("Command returned [%s]\n", t);
#define ERROR_DS "ERROR: unknown DS name"
	    if(!strncmp(t, ERROR_DS, strlen(ERROR_DS))) {
	      char *t2;
	      char rrdid[FILENAME_LEN+1];
	      char new_name[128];
	      int len;
	      if((t = strchr(tmp, '\'')) && (t2 = strchr(&t[1], '\''))) {
		len = (unsigned long)t2 - (unsigned long)t - 1;
		strncpy(new_name, &t[1], len);
		new_name[len] = '\000';
		//printf("name = [%s]\n", new_name);

		s = sensors;
		while(s) {
		  //printf("loop=[%s] s->rrdfile=%p %p\n", s->id, s->rrdfile, r);
		  if(s->rrdfile != r) {
		    s = s->next;
		    continue;
		  }
		  if((save_all_sensors || s->file_found_on_bus) && !s->ignored) {
		    rrd_sensid(s, rrdid);
		    if(!strcmp(rrdid, new_name)) break;
		  }
		  s = s->next;
		}
		if(s) {
		  int rc;
		  printf("Warning: Sensor [%s] [%s] doesn't exist in rrd-file\n", s->id, s->location);
		  if(auto_add_sensors && add_new_sensor_script[0]) {
		    sprintf(cmd, "%s %s %s/%s", add_new_sensor_script, new_name, rrdfile_path, r->name);
		    rc = system(cmd);
		    if(rc == 0) {
		      printf("DS-name [%s] is added to rrd-file\n", rrdid);
		    } else {
		      printf("Failed to add sensor rc=%d cmd=[%s]\n", rc, cmd);
		      // make sure it's ignored in the future if we can't add it
		      s->ignored = 1;
		      generate_templates = 1;
		    }
		  } else {
		    printf("DS-name [%s] is not added automatically. Ignoring it now\n", rrdid);
		    s->ignored = 1;
		    generate_templates = 1;
		  }
		}
	      }
	    }
	  }
	  pclose(fp);
	}
#endif
      } else {
	break;
      }
      r = r->next;
    }
    if(cmd) {
      free(cmd);
      cmd = NULL;
    }
    if(nr_errors >= MAX_ERRORS) {
      // no restart made here...
      nr_errors = 0;
    }
  }
}

