#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <dirent.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#include <db_mini.h>

#define FUSEDEV "/proc/fs/fuse/dev"

#define WWWDIR "/var/httpd"
#define BASEDIR "/data"

#define GREENBAR "http://www.runaware.com/images/gline.gif"
#define REDBAR "http://www.runaware.com/images/rline.gif"
#define BLUEBAR "http://www.runaware.com/images/bline.gif"

#define MAX_ERRORS 5

#define LOG_DIR "/var/httpd/log"
#define ABS(x) ((x) < 0 ? -(x) : (x))
#define SCAN_INTERVAL 60

DB *dbcache;
DB *dblastval;

int save_archive(char *name, char *temp, time_t t)
{
  FILE *fp;
  char filename[128];
  char filename2[128];
  struct stat s;
  char *p;
  int nr, i;
  //char tmp[16];
  int rc;
  sprintf(filename, "%s/%s.log", LOG_DIR, name);

  if(strlen(temp) != 12) {
    printf("Temperature not 12 chars\n");
    return -1;
  }

#define ROTATE_INTERVAL 4

  /* rotate the files */
  if((((t/SCAN_INTERVAL)*SCAN_INTERVAL) % (3600*24*ROTATE_INTERVAL)) == 0) {
    for(i=6; i>=0; i--) {
      sprintf(filename, "%s/%s.log.%d", LOG_DIR, name, i);
      if(i>0)
	sprintf(filename2, "%s/%s.log.%d", LOG_DIR, name, i-1);
      else
	sprintf(filename2, "%s/%s.log", LOG_DIR, name);
      if(stat(filename2, &s) == 0) {
	unlink(filename);
	rename(filename2, filename);
      } else {
	fp = fopen(filename, "w");
	if(fp) fclose(fp);
      }
    }
  }

  sprintf(filename, "%s/%s.log", LOG_DIR, name);
  p = temp;
  while(*p && *p==' ') p++;

  fp = fopen(filename, "a+");
  if(!fp) {
    printf("Cant append to file %s\n", filename);
    fp = fopen(filename, "w");
    if(!fp) {
      printf("Cant create file %s\n", filename);
      return -1;
    }
  }
  
  rc = fprintf(fp, "%ld:%s\n",
	       (((unsigned long)t / SCAN_INTERVAL) * SCAN_INTERVAL), p);
  fclose(fp);

  return 0;
}

int save_file(char *name, char *temp, int html)
{
  FILE *fp;
  int rc;
  char file[128];
  char file_new[128];
  time_t *lt;
  time_t tt = time(NULL);
  DBT key, val;

  key.data = name;
  key.size = strlen(name);
  if(!dblastval->get(dblastval, NULL, &key, &val, 0)) {
    lt = ((time_t *)val.data);
    //printf("name=%s lasttt=%ld\n", name, *lt);
    if((unsigned long)(tt-*lt) >= SCAN_INTERVAL) {
      /* We just save data without dblastval->put() */
      *lt = *lt + SCAN_INTERVAL;
      if(ABS(tt-*lt) > 120) {
	printf("resync time: name=%s tt=%ld\n", name, tt);
	*lt = tt;
      }
      save_archive(name, temp, *lt);
    }
  } else {
    key.data = name;
    key.size = strlen(name);
    val.data = &tt;
    val.size = sizeof(time_t);
    //printf("save first: name=%s tt=%ld\n", name, tt);
    dblastval->put(dblastval, NULL, &key, &val, 0);
    save_archive(name, temp, tt);
  }

  sprintf(file, "%s/%s", WWWDIR, name);
  if(html) {
    strcat(file, ".html");
  } else {
    strcat(file, ".txt");
  }
  sprintf(file_new, "%s.tmp", file);
  fp = fopen(file_new, "w");
  if(!fp) {
    printf("error open %s\n", file_new);
    return -1;
  }
  if(html) {
    rc = fprintf(fp, "<html><body>%s</body></html>", temp);
  } else {
    rc = fprintf(fp, "%s", temp);
  }
  fclose(fp);
  if(!rc) {
    printf("error writing %s\n", file);
  }
  rename(file_new, file);

  key.data = name;
  key.size = strlen(name);
  tt = time(NULL);
  val.data = &tt;
  val.size = sizeof(time_t);
  dbcache->put(dbcache, NULL, &key, &val, 0);

  return 0;
}

int get_temp(char *path, char *fil, char *res)
{
  char filename[128];
  char tmp[80];
  FILE *fp;
  int sz;
  if(!res) return -1;
  *res = '\0';
  
  if(!*path) {
    /* fuse-dev probably not found */
    return -1;
  }

  sprintf(filename, "%s/%s/temperature", path, fil);
  fp = fopen(filename, "r");
  if(!fp) {
    printf("get_temp: fopen(): %s: %s\n", filename, strerror(errno));
    return -errno;
  }
  *tmp = '\0';
  sz = fread(tmp, 1, 20, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("get_temp: fread(): %s %s\n", filename, strerror(errno));
    return -errno;
  }
  tmp[sz] = '\0';
  strcpy(res, tmp);
  return sz;
}

  
int find_fusedev(char *path)
{
  char tmp[128];
  int found = 0;
  char *p, *p2;
  FILE *fp;
  int n;
  fp = fopen("/proc/mounts", "r");
  if(!fp) {
    *path = '\0';
    return found;
  }
  while(!feof(fp)) {
    if(!(p = fgets(tmp, 80, fp))) {
      break;
    }
    if(! strncmp(tmp, FUSEDEV, strlen(FUSEDEV))) {
      found = 1;
      break;
    }
  }
  fclose(fp);

  if(found) {
    while(*p++ != ' ');
    p2 = p;
    n = 0;
    while(*p++ != ' ') n++;
    if(path) {
      strncpy(path, p2, n);
      path[n] = '\0';
    }
  }
  return found;
}

#define OWFSPIDFILE "/var/run/owfs.pid"

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>

void catchchild()
{
  char buf[40];
  pid_t pid;
  int status;
  
  /*signal(SIGCHLD, catchchild);*/ /* Unneeded */
  
  pid = wait4(-1, &status, WUNTRACED, 0);
  if (WIFSTOPPED(status))
    sprintf(buf, "catchchild %d: Child %d stopped\n", getpid(), pid);
  else
    sprintf(buf, "catchchild %d: Child %d died\n", getpid(), pid);
  
  //if (intcrlf) write(STDOUT, "\n", 1);
  //write(STDOUT, buf, strlen(buf));
  //printf(buf);
}

extern char *environ[];

int my_system(char *command)
{
  int pid, status;
  int rc;
  int nr_wait;
  struct sigaction act;

  if (command == NULL)
    return -1;

  pid = vfork();
  if(pid == -1) {
    act.sa_handler = catchchild;
    memset(&act.sa_mask, 0, sizeof(act.sa_mask));
    act.sa_flags = SA_RESTART;
    sigaction(SIGCHLD, &act, NULL);
    return -1;
  }
  if(pid == 0) {
    char *argv[4];
    argv[0] = "sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = 0;
    signal(SIGCHLD, SIG_DFL);
    execve("/bin/sh", argv, environ);
    //execv("/bin/sh", argv);
    _exit(0);
  }

#if 1
  //printf("wait4(%d)\n", pid);
  nr_wait = 0;
  for (;;) {
    int cpid;
#if 1
    cpid = wait4(pid, &status, 0, 0);
    //printf("wait4(%d) cpid=%d\n", cpid);
#else
    cpid = wait4(pid, &status, WNOHANG, 0);
    //printf("wait4(%d) cpid=%d\n", cpid);
    if(cpid == 0) {
      if(nr_wait > 0) {
	kill(pid, SIGTERM);
      }
      nr_wait++;
      sleep(1);
      continue;
    }
#endif
    if ((cpid < 0) && (errno == EINTR))
      continue;
    if (cpid < 0)
      break;
    if (cpid != pid) {
      //fprintf(stderr, "my_system %d: child %d died\n", getpid(), cpid);
      continue;
    }
  }
  //printf("wait4(%d) loop returned\n", pid);

  act.sa_handler = catchchild;
  memset(&act.sa_mask, 0, sizeof(act.sa_mask));
  act.sa_flags = SA_RESTART;
  sigaction(SIGCHLD, &act, NULL);

#else
  do {
    //printf("waitpid(%d)\n", pid);
    rc = waitpid(pid, &status, 0);
    //printf("waitpid(%d) returned %d\n", pid, rc);
    if(rc == -1) {
      if(errno != EINTR) return -1;
    } else {
      return status;
    }
  } while(1);
#endif
}

int restart_owfs(void)
{
  char owfs_cmd[128];
  char umount_cmd[] = "fusermount -u /var/temp";
  FILE *fp;
  int rc, pid;

  sprintf(owfs_cmd, "/usr/bin/owfs -P %s --foreground -d /dev/ttyS1 /var/temp &", OWFSPIDFILE);

  fp = fopen(OWFSPIDFILE, "r");
  if(!fp) {
    printf("No pid-file [%s] found\n", OWFSPIDFILE);
  } else {
    rc = fscanf(fp, "%d", &pid);
    fclose(fp);
    if(rc != 1) {
      printf("Can't find pid in file.\n");
      return -1;
    }
    rc = kill(pid, SIGTERM);
    if(rc < 0) {
      printf("Can't kill process %d (%s)\n", pid, strerror(errno));
    }
    sleep(1);
  }
  if(find_fusedev(NULL)) {
    //printf("Unmount fuse-dir\n");
    rc = my_system(umount_cmd);
    if(rc < 0) {
      printf("Error from fusermount (%s)\n", strerror(errno));
    }
    //printf("done.\n");
    sleep(1);
  }
  //printf("Restart owfs.\n");
  rc = my_system(owfs_cmd);
  if(rc < 0) {
    printf("Error from owfs (%s)\n", strerror(errno));
  }
  //printf("done.\n");
  sleep(1);
  return rc;
}


int main(int argc, char **argv)
{
  FILE *fp;
  char *p, *p2;
  char tmp[128];
  char device_path[80];
  char filename[128];
  int found_fuse = 0;
  struct dirent **namelist;
  float lenf;
  FILE *fp2;
  struct stat st;
  struct tm *t;
  int n, len, rc;
  time_t tnow;
  int last_scan, scan_interval;
  int nr_errors, nr_sensors;

  struct sigaction act;

  act.sa_handler = catchchild;
  memset(&act.sa_mask, 0, sizeof(act.sa_mask));
  act.sa_flags = SA_RESTART;
  sigaction(SIGCHLD, &act, NULL);

  last_scan = 0;
  scan_interval = SCAN_INTERVAL;

  if(db_create(&dbcache, NULL, 0) ||
     dbcache->open(dbcache, NULL, NULL, NULL, DB_HASH, DB_CREATE, 0)) {
    printf("Error create db\n");
    exit(0);
  }

  if(db_create(&dblastval, NULL, 0) ||
     dblastval->open(dblastval, NULL, NULL, NULL, DB_HASH, DB_CREATE, 0)) {
    printf("Error create db\n");
    exit(0);
  }

  nr_errors = 0;
  while(1) {
    found_fuse = find_fusedev(device_path);
    if(!found_fuse) {
      nr_errors++;
    } else if(time(NULL) > (last_scan + SCAN_INTERVAL)) {
      last_scan = time(NULL);
      nr_sensors = 0;
      n = scandir(device_path, &namelist, 0, alphasort);
      if(n < 0) {
	printf("error scandir: %s\n", strerror(errno));
	nr_errors++;
	if(errno == ENOTCONN) nr_errors = MAX_ERRORS;
      } else {
	while(n--) {
	  if(nr_errors < MAX_ERRORS) {
	    //printf("%s\n", namelist[n]->d_name);
	    /* only read temp sensors */
	    if(!strncmp(namelist[n]->d_name, "10.", 3)) {
	      if((rc = get_temp(device_path, namelist[n]->d_name, tmp)) < 0) {
		printf("Error reading: %s\n", namelist[n]->d_name);
		if(rc == -ENOTCONN) nr_errors = MAX_ERRORS;
	      } else {
		nr_errors = 0;
		nr_sensors++;
		save_file(namelist[n]->d_name, tmp, 0);
	      }
	      //printf("done\n");
	    }
	  }
	  free(namelist[n]);
	}
	free(namelist);
	if(nr_sensors == 0) {
	  nr_errors++;
	}
	{
	  FILE *fp = fopen("/tmp/restart", "r");
	  if(fp) {
	    fclose(fp);
	    nr_errors = MAX_ERRORS;
	    unlink("/tmp/restart");
	  }
	}
      }
    } else {
      struct db_ent *e;
      DBT *k;
      time_t t;
      time_t tnow;
      char file[128];
      e = dbcache->head->next;

      //printf("update all in cache\n");

      while(e) {
	tnow = time(NULL);
	k = e->key;
	memcpy(file, k->data, k->size);
	file[k->size] = '\0';
	k = e->val;
	if(k->size != sizeof(time_t)) {
	  printf("error size is %d\n", k->size);
	} else {
	  memcpy(&t, k->data, k->size);
	}
	//printf("found %s  (%d ago)\n", file, tnow-t);

	if((tnow-t) > 10) {
	  DBT key;
	  if((rc = get_temp(device_path, file, tmp)) < 0) {
	    key.data = file;
	    key.size = strlen(file);
	    e = e->next; /* have to find next entry before freeing memory */
	    printf("Error updating: %s (deleted)\n", file);
	    dbcache->del(dbcache, NULL, &key, 0);
	    dblastval->del(dblastval, NULL, &key, 0);
	  } else {
	    //printf("update %s (old %d)\n", file, tnow-t);
	    save_file(file, tmp, 0);
	    e = e->next;
	  }
	} else {
	  e = e->next;
	}
      }
    }

    if(nr_errors >= MAX_ERRORS) {
      syslog(LOG_WARNING,"No sensors %d times in a row. Restarting owfs.\n", nr_errors);
      //printf("No sensors %d times in a row. Restarting owfs.\n", nr_errors);
      if(restart_owfs() < 0) {
	//printf("Failed to restart owfs.\n");
	syslog(LOG_WARNING,"Failed to restart owfs.\n");
      }
      nr_errors = 0;
    }
    
    sprintf(tmp, "%s/index.html.new", WWWDIR);
    fp = fopen(tmp, "w");
    if(!fp) {
      sleep(5);
      continue;
    }

    fprintf(fp, "<html><body><h1>Current temperatures</h1>\n");
    fprintf(fp, "<table width=\"700\" cellpadding=\"0\" cellspacing=\"0\" border=\"0\">\n");
    
    fprintf(fp, "<tr><td width=\"100\">Sensor ID\n");
    fprintf(fp, "</td><td width=\"70\">Degrees\n");
    fprintf(fp, "</td><td width=\"150\">Last update\n");
    fprintf(fp, "</td><td width=\"300\">\n");

    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, BLUEBAR);
    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, REDBAR);
    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, BLUEBAR);
    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, REDBAR);
    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, BLUEBAR);
    fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">", 50, REDBAR);
    fprintf(fp, "</td></tr>\n");
    
    n = scandir(WWWDIR, &namelist, 0, alphasort);
    if(n < 0) {
      printf("error scandir: %s\n", strerror(errno));
    } else {
      while(n--) {
	p = namelist[n]->d_name;
	len = strlen(p);
	if((len > 5) && strcmp(p, "index.html") &&
	   //(!strncmp(&p[len-5], ".html", 5) ||
	   !strncmp(&p[len-4], ".txt", 4)
	   ) {
	  fprintf(fp, "<tr><td width=\"100\">\n");
	  fprintf(fp, "<a href=\"%s/%s\">%s</a><br>\n", BASEDIR, p, p);
	  sprintf(filename, "%s/%s", WWWDIR, p);
	  lenf = (float)0.0;
	  len = 1;
	  fp2 = fopen(filename, "r");
	  if(fp2) {
	    if(fscanf(fp2, "%f", &lenf) > 0) {
	      len = (int)(lenf * 10.0);
	    }
	    fclose(fp2);
	  }
	  fprintf(fp, "</td><td width=\"70\">%7.4f\n", lenf);

	  if(!stat(filename, &st)) {
	    t = localtime(&st.st_mtime);
	    rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", t);
	    fprintf(fp, "</td><td width=\"150\"><font size=\"-1\">%s</font>\n", tmp);
	  }

	  fprintf(fp, "</td><td width=\"300\">\n");
	  fprintf(fp, "<img border=\"0\" height=\"16\" width=\"%d\" src=\"%s\">\n", len, BLUEBAR);
	  fprintf(fp, "</td></tr>\n");
	}
	free(namelist[n]);
      }
      free(namelist);
    }
    fprintf(fp, "</table>\n");

    if(!found_fuse) {
      fprintf(fp, "<br>(fuse-dev not found, and therefor not values)<br>\n");
    }

    tnow = time(NULL);
    t = localtime(&tnow);
    rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", t);
    fprintf(fp, "<br><hr>Last modified: %s<br>\n", tmp);
    fprintf(fp, "&copy; Christian Magnusson, mag (at) mag.cx<br>\n");

    fprintf(fp, "</body></html>\n");
    fclose(fp);

    sprintf(tmp, "%s/index.html.new", WWWDIR);
    sprintf(filename, "%s/index.html", WWWDIR);
    rename(tmp, filename);

    sleep(SCAN_INTERVAL/3);
  }
  dbcache->close(dbcache, 0);
}

